const resources = $.resources || [];
log(JSON.stringify(resources, null, 2));
const resource = resources.find((r) => r.key === 'json');
const executions = resource.content || [];

//
// executions example:
// [
//     {
//         object: 'job_execution',
//         jobId: 'xx',
//         id: 'xx',
//         pipelineId: 'xx',
//         createdAt: '2026-02-11T11:12:12.618Z',
//         updatedAt: '2026-02-11T11:12:12.609Z',
//         endedAt: '',
//         input: {
//             context: {
//                 jobAccountId: 'xx',
//                 jobId: 'xx',
//                 userShardId: 'xx',
//                 userAccountId: 'xx',
//                 userId: 'xx',
//             },
//         },
//         startedAt: '2026-02-11T11:12:12.590Z',
//         status: 'COMPLETED',
//         output: {},
//         info: {
//             title: 'xx',
//         },
//     },
// ];

const escapeXml = (str) => {
    if (!str) return '';
    return String(str).replace(
        /[<>&"']/g,
        (c) =>
            ({
                '<': '&lt;',
                '>': '&gt;',
                '&': '&amp;',
                '"': '&quot;',
                "'": '&apos;',
            }[c])
    );
};

const formatDate = (date) => {
    if (!date) return '';
    const d = new Date(date);
    return d.toISOString().replace('T', ' ').substring(0, 19);
};

// Generate cells for the Excel sheet
let cells = [];
let line = 1;

// Headers
cells.push(`<Cell-Text line="${line}" column="1">Job ID</Cell-Text>`);
cells.push(`<Cell-Text line="${line}" column="2">Job Name</Cell-Text>`);
cells.push(`<Cell-Text line="${line}" column="3">Status</Cell-Text>`);
cells.push(`<Cell-Text line="${line}" column="4">Started At</Cell-Text>`);
cells.push(`<Cell-Text line="${line}" column="5">Ended At</Cell-Text>`);
cells.push(
    `<Cell-Text line="${line}" column="6">Duration (seconds)</Cell-Text>`
);
cells.push(
    `<Cell-Text line="${line}" column="7">Created By (user id)</Cell-Text>`
);

// Data rows
executions.forEach((exec) => {
    line++;

    const duration =
        exec.endedAt && exec.startedAt
            ? Math.round(
                  (new Date(exec.endedAt) - new Date(exec.startedAt)) / 1000
              )
            : null;
    cells.push(
        `<Cell-Text line="${line}" column="1">${escapeXml(
            exec.jobId || 'N/A'
        )}</Cell-Text>`
    );
    cells.push(
        `<Cell-Text line="${line}" column="2">${escapeXml(
            exec.info?.title || 'N/A'
        )}</Cell-Text>`
    );
    cells.push(
        `<Cell-Text line="${line}" column="3">${escapeXml(
            exec.status
        )}</Cell-Text>`
    );
    cells.push(
        `<Cell-Text line="${line}" column="4">${formatDate(
            exec.startedAt
        )}</Cell-Text>`
    );
    cells.push(
        `<Cell-Text line="${line}" column="5">${formatDate(
            exec.endedAt
        )}</Cell-Text>`
    );

    if (duration !== null) {
        cells.push(
            `<Cell-Number line="${line}" column="6">${duration}</Cell-Number>`
        );
    } else {
        cells.push(`<Cell-Text line="${line}" column="6"></Cell-Text>`);
    }

    cells.push(
        `<Cell-Text line="${line}" column="7">${escapeXml(
            exec.input?.context?.userId || ''
        )}</Cell-Text>`
    );
});

// Generate the XML in the format expected by file-generation-xlsx
const xml = `<?xml version="1.0" encoding="UTF-8"?>
<Generate-Excel>
  <File>
    <File-Name>job_executions_report.xlsx</File-Name>
    <Template-Key>report</Template-Key>
    <Sheets>
      <Sheet>
        <Sheet-Name>Job Executions</Sheet-Name>
        <Cells>
          ${cells.join('\n          ')}
        </Cells>
      </Sheet>
    </Sheets>
  </File>
</Generate-Excel>`;

return xml;
